<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

/**
 * Elementor blog-post Widget.
 *
 * Elementor widget that uses the blog-post control.
 *
 * @since 1.0.0
 */
class Elementor_Blog_Post_Widget extends \Elementor\Widget_Base {
	/**
	 * Get widget name.
	 *
	 * Retrieve blog-post widget name.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget name.
	 */
	public function get_name() {
		return 'blog-post';
	}
	/**
	 * Get widget title.
	 *
	 * Retrieve blog-post widget title.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget title.
	 */
	public function get_title() {
		return esc_html__( 'Lawfinity Blog Posts', 'elementor-blog-post-control' );
	}

	/**
	 * Get widget icon.
	 *
	 * Retrieve blog-post widget icon.
	 *
	 * @since 1.0.0
	 * @access public
	 * @return string Widget icon.
	 */
	// public function get_icon() {
	// 	return 'eicon-carousel-loop';
	// }

	/**
	 * Register blog-post widget controls.
	 *
	 * Add input fields to allow the user to customize the widget settings.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function register_controls() {
        
        $this->start_controls_section(
			'content_section',
			[
				'label' => esc_html__( 'Content', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);
		$this->add_control(
			'number_of_post_blogs',
			[
				'label' => __( 'Number of Posts', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::NUMBER,
				'default' => 3, // Set a default value
				'min' => 1,
				'max' => 21,
				'step' => 1,
			]
			);
		$this->add_control(
			'selected_style',
			[
				'label' => esc_html__( 'Select Style', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'style1', // Set the default style
				'options' => [
					'style1' => esc_html__( 'Style 1', 'elementor-blog-post-control' ),
					'style2' => esc_html__( 'Style 2', 'elementor-blog-post-control' ),
					'style3' => esc_html__( 'Style 3', 'elementor-blog-post-control' ),
				],
			]
		);
		$this->add_control(
			'xtremefitness_post_blog_category',
			[
				'label' => esc_html__( 'Select Post Category', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT2,
				'multiple' => true,
				'options' => $this->get_post_blog_categories(),
				'default' => 'post',
			]
		);
		$this->add_control(
			'button_text',
			[
				'label' => esc_html__( 'Button Text', 'elementor-immersive-button-control' ),
				'label_block' => true,
				'placeholder' => __( 'Read More', 'elementor-immersive-button-control' ),
				'type' => 'text',
			]
		);
        $this->end_controls_section();
		$this->start_controls_section(
			'post_order_section',
			[
				'label' => esc_html__('Post Order', 'elementor-blog-post-control'),
				'tab'   => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);
		
		$this->add_control(
			'post_order',
			[
				'label'   => esc_html__('Post Order', 'elementor-blog-post-control'),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'asc'  => [
						'title' => esc_html__('Ascending', 'elementor-blog-post-control'),
						'icon'  => 'fa fa-sort-amount-asc',
					],
					'desc' => [
						'title' => esc_html__('Descending', 'elementor-blog-post-control'),
						'icon'  => 'fa fa-sort-amount-desc',
					],
				],
				'default' => 'asc',
			]
		);
		
		$this->add_control(
			'post_order_by',
			[
				'label'   => esc_html__('Sort Order', 'elementor-blog-post-control'),
				'type'    => \Elementor\Controls_Manager::CHOOSE,
				'options' => [
					'date'  => [
						'title' => esc_html__('Latest Posts', 'elementor-blog-post-control'),
						'icon'  => 'fa fa-calendar',
					],
					'title' => [
						'title' => esc_html__('Alphabetical Order', 'elementor-blog-post-control'),
						'icon'  => 'fa fa-sort-alpha-asc',
					],
				],
				'default' => 'date',
			]
		);
		
		$this->end_controls_section();
		$this->start_controls_section(
			'image_section',
			[
				'label' => esc_html__( 'Image Controls', 'elementor-blog-post-control' ),
				'tab' => \Elementor\Controls_Manager::TAB_CONTENT,
			]
		);
		
		$this->add_responsive_control(
			'image_size',
			[
				'label' => esc_html__( 'Image Size', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'default' => 'medium', // Set your default image size
				'options' => [
					'thumbnail' => 'Thumbnail',
					'medium' => 'Medium',
					'large' => 'Large',
					'full' => 'Full',
				],
			]
		);
		
		$this->end_controls_section();
		$this->start_controls_section(
			'image_control_section',
			[
				'label' => esc_html__( 'Image Dimensions', 'elementor-blog-post-control' ),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
			'image_width',
			[
				'label' => esc_html__( 'Image Width', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} img' => 'width: {{SIZE}}{{UNIT}};',
				],
			]);
		
		$this->add_responsive_control(
			'image_height',
			[
				'label' => esc_html__( 'image Height', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} img' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'object-fit',
			[
				'label' => esc_html__( 'Object Fit', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'condition' => [
					'image_height[size]!' => '',
				],
				'options' => [
					'' => esc_html__( 'Default', 'elementor-blog-post-control' ),
					'fill' => esc_html__( 'Fill', 'elementor-blog-post-control' ),
					'cover' => esc_html__( 'Cover', 'elementor-blog-post-control' ),
					'contain' => esc_html__( 'Contain', 'elementor-blog-post-control' ),
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} img' => 'object-fit: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'object-position',
			[
				'label' => esc_html__( 'Object Position', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'center center' => esc_html__( 'Center Center', 'elementor-blog-post-control' ),
					'center left' => esc_html__( 'Center Left', 'elementor-blog-post-control' ),
					'center right' => esc_html__( 'Center Right', 'elementor-blog-post-control' ),
					'top center' => esc_html__( 'Top Center', 'elementor-blog-post-control' ),
					'top left' => esc_html__( 'Top Left', 'elementor-blog-post-control' ),
					'top right' => esc_html__( 'Top Right', 'elementor-blog-post-control' ),
					'bottom center' => esc_html__( 'Bottom Center', 'elementor-blog-post-control' ),
					'bottom left' => esc_html__( 'Bottom Left', 'elementor-blog-post-control' ),
					'bottom right' => esc_html__( 'Bottom Right', 'elementor-blog-post-control' ),
				],
				'default' => 'center center',
				'selectors' => [
					'{{WRAPPER}} img' => 'object-position: {{VALUE}};',
				],
				'condition' => [
					'object-fit' => 'cover',
				],
			]
		);
		
		$this->end_controls_section();		
		$this->start_controls_section(
			'image_control_section1',
			[
				'label' => esc_html__( 'Figure Dimensions', 'elementor-blog-post-control' ),
				'tab' => \Elementor\Controls_Manager::TAB_STYLE,
			]
		);
		
		$this->add_responsive_control(
			'image_width1',
			[
				'label' => esc_html__( 'Image Width', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} figure' => 'width: {{SIZE}}{{UNIT}};',
				],
			]);
		
		$this->add_responsive_control(
			'image_height1',
			[
				'label' => esc_html__( 'image Height', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SLIDER,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} figure' => 'height: {{SIZE}}{{UNIT}};',
				],
			]
		);
		$this->add_responsive_control(
			'object-fit1',
			[
				'label' => esc_html__( 'Object Fit', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'condition' => [
					'image_height[size]!' => '',
				],
				'options' => [
					'' => esc_html__( 'Default', 'elementor-blog-post-control' ),
					'fill' => esc_html__( 'Fill', 'elementor-blog-post-control' ),
					'cover' => esc_html__( 'Cover', 'elementor-blog-post-control' ),
					'contain' => esc_html__( 'Contain', 'elementor-blog-post-control' ),
				],
				'default' => '',
				'selectors' => [
					'{{WRAPPER}} figure' => 'object-fit: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'object-position1',
			[
				'label' => esc_html__( 'Object Position', 'elementor-blog-post-control' ),
				'type' => \Elementor\Controls_Manager::SELECT,
				'options' => [
					'center center' => esc_html__( 'Center Center', 'elementor-blog-post-control' ),
					'center left' => esc_html__( 'Center Left', 'elementor-blog-post-control' ),
					'center right' => esc_html__( 'Center Right', 'elementor-blog-post-control' ),
					'top center' => esc_html__( 'Top Center', 'elementor-blog-post-control' ),
					'top left' => esc_html__( 'Top Left', 'elementor-blog-post-control' ),
					'top right' => esc_html__( 'Top Right', 'elementor-blog-post-control' ),
					'bottom center' => esc_html__( 'Bottom Center', 'elementor-blog-post-control' ),
					'bottom left' => esc_html__( 'Bottom Left', 'elementor-blog-post-control' ),
					'bottom right' => esc_html__( 'Bottom Right', 'elementor-blog-post-control' ),
				],
				'default' => 'center center',
				'selectors' => [
					'{{WRAPPER}} img' => 'object-position: {{VALUE}};',
				],
				'condition' => [
					'object-fit' => 'cover',
				],
			]
		);
		
		$this->end_controls_section();	
        
	}
		/**
	 * Get blog-post categories.
	 *
	 * Retrieve the blog-post categories to populate the category control.
	 *
	 * @since 1.0.0
	 * @access protected
	 * @return array blog-post categories.
	 */
	protected function get_post_blog_categories() {
		$categories = get_categories( array( 'taxonomy' => 'category' ) );
		$options = array();

		foreach ( $categories as $category ) {
			$options[ $category->term_id ] = $category->name;
		}
		return $options;
	}
	/**
	 * Render blog-post widget output on the frontend.
	 *
	 * Written in PHP and used to generate the final HTML.
	 *
	 * @since 1.0.0
	 * @access protected
	 */
	protected function render() {
		$settings = $this->get_settings_for_display();
		$number_of_post_blogs = $settings['number_of_post_blogs'];
		$button_text = $settings['button_text'];
		$category = $settings['xtremefitness_post_blog_category'];
		$image_size = $settings['image_size']; // Get the selected image size
	
		// Get user-selected post order (ascending or descending)
		$post_order = ($settings['post_order'] === 'asc') ? 'ASC' : 'DESC';
	
		// Get user-selected sorting order (latest posts or alphabetical)
		$post_order_by = ($settings['post_order_by'] === 'date') ? 'date' : 'title';
		$selected_style = $settings['selected_style'];
	
		$args = array(
			'post_type'      => 'post',
			'posts_per_page' => $number_of_post_blogs,
			'order'          => $post_order,    // Use user-selected post order
			'orderby'        => $post_order_by, // Use user-selected sorting order
		);
	
		// Check if a specific category is selected
		if (!empty($category) && $category != 'post') {
			$args['tax_query'] = array(
				array(
					'taxonomy' => 'category',
					'field'    => 'Name',
					'terms'    => $category,
				),
			);
		}
	
		$result = new WP_Query($args);
	
		if ($selected_style === 'style1') { ?>
			<section class="article-con">
				<div class="container">
					<div class="row" data-aos="fade-up">
						<div class="owl-carousel owl-theme">
							<?php if ($result->have_posts()) {
								while ($result->have_posts()) {
									$result->the_post();
									$url = wp_get_attachment_url(get_post_thumbnail_id(), "post_thumbnail");
									$meta = get_post_meta(get_the_ID(), 'post_thumbnail', true);
									$post_id = get_the_ID(); 
									$comment_count = get_comments_number($post_id);
								?>
								<div class="item">
									<div class="article-box">
										<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none"><figure class="mb-0">
											<img src="<?php echo $url ?>" alt="" class="img-fluid">
										</figure></a>
										<div class="box-content">
											<div class="span_wrapper">
												<span><?php echo get_the_date('M d, Y'); ?></span>
												<span class="dash">-</span>
												<span><?php echo $comment_count ?> Comments</span>
											</div>
											<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none">
												<h6><?php the_title(); ?></h6>
											</a>
											<p class="text-size-14"><?php echo wp_trim_words(get_the_content(), 9) ?></p>
											<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none">
												<i class="fa-solid fa-arrow-right"></i>
											</a>
										</div>
									</div>
								</div>
							<?php }
							}
							?>
						</div>
					</div>
				</div>
			</section>
		<?php } elseif ($selected_style === 'style2') { ?>
			<section class="blog-con">
				<div class="container">
					<div class="row" data-aos="fade-up">
				<?php if ($result->have_posts()) {
					while ($result->have_posts()) {
						$result->the_post();
						$url = wp_get_attachment_url(get_post_thumbnail_id(), "post_thumbnail");
						$meta = get_post_meta(get_the_ID(), 'post_thumbnail', true);
						$post_id = get_the_ID(); 
						$comment_count = get_comments_number($post_id);
					?>
					<div class="col-lg-4 col-md-4 col-sm-12 col-12">
						<div class="article-box">
						<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none">
							<figure class="mb-0">
								<img src="<?php echo $url ?>" alt="" class="img-fluid">
							</figure>
							<div class="box-content">
								<div class="span_wrapper">
									<span><?php echo get_the_date('M d, Y'); ?></span>
									<span class="dash">-</span>
									<span><?php echo $comment_count ?> Comments</span>
								</div>
								<a href="<?php echo get_post_permalink(); ?>" class="text-decoration-none">
									<h6><?php the_title(); ?></h6>
								</a>
								<p class="text-size-14 mb-0"><?php echo wp_trim_words(get_the_content(), 9) ?></p>
							</div>
						</a>
						</div>
					</div>
				<?php }
				}
				?>
					</div>
				</div>
			</section>
		<?php } ?>
	<?php
	}
	
}
